// ===========================================
// CONFIGURATION
// ===========================================
const WEBSITE_URL = 'https://perfume-compare.com';
// ===========================================

// Security: HTML escape to prevent XSS
function escapeHtml(text) {
  if (!text) return '';
  const div = document.createElement('div');
  div.textContent = String(text);
  return div.innerHTML;
}

document.getElementById('scan-btn').addEventListener('click', async () => {
  const content = document.getElementById('content');
  
  content.innerHTML = `
    <div class="state">
      <div class="spinner"></div>
      <div class="state-title">Scanne...</div>
      <div class="state-text" id="status">Suche Angebote...</div>
      <div class="progress"><div class="progress-bar" id="progress" style="width: 0%"></div></div>
    </div>
  `;
  
  try {
    const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
    
    // Check if on Parfumo
    if (!tab.url.includes('parfumo.de') && !tab.url.includes('parfumo.com')) {
      showError('Nicht auf Parfumo', 'Bitte gehe zuerst zu parfumo.de und suche im Souk.');
      return;
    }
    
    // Check if on Souk
    if (!tab.url.includes('s_souk.php') && !tab.url.includes('/Souk')) {
      showError('Nicht im Souk', 'Bitte suche zuerst im Souk nach einem Parfum.<br><br>Gehe zu: Souk → Suche ein Parfum');
      return;
    }
    
    // Get listing URLs from the current page
    const urlResults = await chrome.scripting.executeScript({
      target: { tabId: tab.id },
      func: getListingUrls
    });

    let urls = urlResults[0].result || [];

    // Check for pagination
    const paginationResult = await chrome.scripting.executeScript({
      target: { tabId: tab.id },
      func: getPaginationInfo
    });

    const pagination = paginationResult[0].result;

    if (pagination && pagination.totalPages > 1) {
      // Use the detected page parameter (current_page for Parfumo)
      const pageParam = pagination.pageParam || 'current_page';

      // Construct base URL (remove any existing page parameter)
      const baseUrl = tab.url
        .replace(/&current_page=\d+/gi, '')
        .replace(/\?current_page=\d+&?/gi, '?')
        .replace(/&p=\d+/gi, '')
        .replace(/\?p=\d+&?/gi, '?')
        .replace(/\?$/, '');
      const hasParams = baseUrl.includes('?');

      for (let page = 2; page <= pagination.totalPages; page++) {
        const pageUrl = hasParams ? `${baseUrl}&${pageParam}=${page}` : `${baseUrl}?${pageParam}=${page}`;

        document.getElementById('status').textContent = `Seite ${page} von ${pagination.totalPages}...`;
        document.getElementById('progress').style.width = `${((page - 1) / pagination.totalPages) * 30}%`;

        try {
          const pageUrlsResult = await chrome.scripting.executeScript({
            target: { tabId: tab.id },
            func: fetchPageUrls,
            args: [pageUrl]
          });

          const pageUrls = pageUrlsResult[0].result || [];
          if (pageUrls.length > 0) {
            urls = [...urls, ...pageUrls];
          }
        } catch (e) {
          console.error('Pagination error:', e);
        }
      }

      // Remove duplicates
      urls = [...new Set(urls)];
    }

    if (!urls || urls.length === 0) {
      showError('Keine Angebote', 'Keine Souk-Angebote auf dieser Seite gefunden.');
      return;
    }

    document.getElementById('status').textContent = `${urls.length} Angebote gefunden. Lade Details...`;
    
    // Fetch live exchange rates first
    let exchangeRates = null;
    try {
      const ratesResult = await chrome.scripting.executeScript({
        target: { tabId: tab.id },
        func: fetchExchangeRates
      });
      exchangeRates = ratesResult[0].result;
      console.log('Fetched live exchange rates:', exchangeRates);
    } catch (e) {
      console.log('Could not fetch live rates, using fallback');
    }
    
    // Fetch each listing with progress updates
    const listings = [];
    const progressOffset = pagination && pagination.totalPages > 1 ? 30 : 0; // 30% was used for pagination
    for (let i = 0; i < urls.length; i++) {
      document.getElementById('status').textContent = `Lade Angebot ${i + 1} von ${urls.length}...`;
      document.getElementById('progress').style.width = `${progressOffset + ((i + 1) / urls.length) * (100 - progressOffset)}%`;
      
      const result = await chrome.scripting.executeScript({
        target: { tabId: tab.id },
        func: fetchListing,
        args: [urls[i], exchangeRates]
      });
      
      if (result[0].result) {
        listings.push(result[0].result);
      }
    }
    
    if (listings.length === 0) {
      showError('Keine Daten', 'Konnte keine Preisdaten extrahieren.<br><br>Stelle sicher, dass du eingeloggt bist.');
      return;
    }
    
    // Sort by price per ml (EUR), sold items at the bottom
    listings.sort((a, b) => {
      // Sold items go to bottom
      if (a.sold && !b.sold) return 1;
      if (!a.sold && b.sold) return -1;
      // No price goes to bottom (but above sold)
      if (!a.pricePerMlEur && b.pricePerMlEur) return 1;
      if (a.pricePerMlEur && !b.pricePerMlEur) return -1;
      if (!a.pricePerMlEur && !b.pricePerMlEur) return 0;
      // Sort by price per ml
      return a.pricePerMlEur - b.pricePerMlEur;
    });
    
    // Compress and encode data for shorter URLs
    const jsonData = JSON.stringify(listings);
    const compressed = LZString.compressToEncodedURIComponent(jsonData);
    const resultUrl = `${WEBSITE_URL}/results.html#${compressed}`;

    // Auto-redirect to results page
    chrome.tabs.create({ url: resultUrl });
    
  } catch (error) {
    console.error('Error:', error);
    // Escape error message to prevent XSS
    showError('Fehler', 'Etwas ist schief gelaufen.<br><br>' + escapeHtml(error.message));
  }
});

function showError(title, message) {
  document.getElementById('content').innerHTML = `
    <div class="state">
      <div class="state-icon">😕</div>
      <div class="state-title">${title}</div>
      <div class="state-text">${message}</div>
      <button class="btn" id="retry-btn" style="margin-top: 16px;">Erneut versuchen</button>
    </div>
  `;
  document.getElementById('retry-btn').addEventListener('click', () => location.reload());
}

// ============================================
// Functions that run IN the Parfumo page
// ============================================

function getListingUrls() {
  // Find all Souk item links on current page
  const links = document.querySelectorAll('a[href*="/Souk/Item/"]');
  const urls = [...new Set([...links].map(a => a.href))];
  return urls;
}

function getPaginationInfo() {
  // Look for pagination text like "1 - 20 von 72" or "1 - 20 of 72"
  const paginationText = document.body.innerText.match(/(\d+)\s*[-–]\s*(\d+)\s*(?:von|of)\s*(\d+)/i);
  if (paginationText) {
    const start = parseInt(paginationText[1]);
    const end = parseInt(paginationText[2]);
    const total = parseInt(paginationText[3]);
    const perPage = end - start + 1;
    const totalPages = Math.ceil(total / perPage);

    // Find pagination links to detect the URL pattern
    // Look for links with page numbers in the pagination area
    const allLinks = document.querySelectorAll('a[href]');
    let pageParam = null;
    let pageUrlTemplate = null;

    for (const link of allLinks) {
      const href = link.href;
      // Look for links that have page numbers - include current_page which Parfumo uses
      const pageMatch = href.match(/[?&](current_page|p|page|seite)=(\d+)/i);
      if (pageMatch && parseInt(pageMatch[2]) > 1) {
        pageParam = pageMatch[1];
        // Store a template URL (replace the page number with a placeholder)
        pageUrlTemplate = href.replace(/([?&])(current_page|p|page|seite)=\d+/i, '$1$2={PAGE}');
        break;
      }
    }

    return {
      start, end, total, perPage, totalPages,
      currentPage: Math.ceil(start / perPage),
      pageParam: pageParam || 'current_page',  // Default to current_page for Parfumo
      pageUrlTemplate
    };
  }
  return null;
}

async function fetchPageUrls(pageUrl) {
  // Fetch a page and extract listing URLs
  console.log('[Pagination] Fetching:', pageUrl);
  try {
    const response = await fetch(pageUrl, { credentials: 'include' });
    console.log('[Pagination] Response status:', response.status);
    const html = await response.text();
    console.log('[Pagination] HTML length:', html.length);

    const parser = new DOMParser();
    const doc = parser.parseFromString(html, 'text/html');

    // Try multiple selectors
    let links = doc.querySelectorAll('a[href*="/Souk/Item/"]');
    console.log('[Pagination] Links with /Souk/Item/:', links.length);

    // Also try other patterns
    if (links.length === 0) {
      links = doc.querySelectorAll('a[href*="Souk"][href*="Item"]');
      console.log('[Pagination] Links with Souk+Item:', links.length);
    }

    // Debug: show first few hrefs found
    const allLinks = doc.querySelectorAll('a[href]');
    console.log('[Pagination] Total links on page:', allLinks.length);
    const sampleHrefs = [...allLinks].slice(0, 10).map(a => a.getAttribute('href'));
    console.log('[Pagination] Sample hrefs:', sampleHrefs);

    // Get base URL for resolving relative URLs
    const urlObj = new URL(pageUrl);
    const baseUrl = `${urlObj.protocol}//${urlObj.host}`;

    // Extract and resolve URLs
    const urls = [...links].map(a => {
      const href = a.getAttribute('href');
      if (href.startsWith('http')) return href;
      if (href.startsWith('/')) return baseUrl + href;
      return baseUrl + '/' + href;
    });
    console.log('[Pagination] Extracted URLs:', urls.length);

    return [...new Set(urls)];
  } catch (e) {
    console.error('fetchPageUrls error:', e);
    return [];
  }
}

// Fetch live exchange rates from API
async function fetchExchangeRates() {
  try {
    // Using exchangerate-api.com (free, no API key needed for basic use)
    const response = await fetch('https://api.exchangerate-api.com/v4/latest/EUR');
    const data = await response.json();
    // We need rates TO EUR, so we invert
    const rates = {};
    for (const [currency, rate] of Object.entries(data.rates)) {
      rates[currency] = 1 / rate; // Convert FROM currency TO EUR
    }
    rates['EUR'] = 1;
    rates['€'] = 1;
    return rates;
  } catch (e) {
    console.error('Failed to fetch exchange rates:', e);
    return null;
  }
}

async function fetchListing(url, liveRates) {
  // Fallback currency conversion rates to EUR (if live rates unavailable)
  const fallbackRates = {
    'EUR': 1, '€': 1, 'EURO': 1,
    'CHF': 0.94,    // Swiss Franc
    'GBP': 1.18,    // British Pound
    '£': 1.18,
    'USD': 0.92,    // US Dollar
    '$': 0.92,
    'PLN': 0.23,    // Polish Zloty
    'SEK': 0.088,   // Swedish Krona
    'DKK': 0.13,    // Danish Krone
    'NOK': 0.086,   // Norwegian Krone
    'CZK': 0.040,   // Czech Koruna
    'HUF': 0.0025,  // Hungarian Forint
  };
  
  // Use live rates if available, otherwise fallback
  const currencyRates = liveRates || fallbackRates;
  
  // Helper to parse numbers
  function parseNum(str) {
    if (!str) return null;
    let cleaned = str.replace(/[^\d,.-]/g, '');
    cleaned = cleaned.replace(/\./g, '').replace(',', '.');
    const num = parseFloat(cleaned);
    return isNaN(num) ? null : num;
  }
  
  // Helper to detect currency and extract price
  // DEFAULT: If no currency symbol found, assume EUR
  function parsePriceWithCurrency(text) {
    if (!text) return { price: null, currency: 'EUR' };
    
    // Clean the text
    const cleanText = text.trim();
    
    // Patterns for different currencies (order matters - specific first)
    const patterns = [
      { regex: /(\d+(?:[.,]\d+)?)\s*CHF/i, currency: 'CHF' },
      { regex: /CHF\s*(\d+(?:[.,]\d+)?)/i, currency: 'CHF' },
      { regex: /(\d+(?:[.,]\d+)?)\s*Fr\.?(?:\s|$)/i, currency: 'CHF' },
      { regex: /(\d+(?:[.,]\d+)?)\s*SFr/i, currency: 'CHF' },
      { regex: /(\d+(?:[.,]\d+)?)\s*£/, currency: 'GBP' },
      { regex: /£\s*(\d+(?:[.,]\d+)?)/, currency: 'GBP' },
      { regex: /(\d+(?:[.,]\d+)?)\s*GBP/i, currency: 'GBP' },
      { regex: /(\d+(?:[.,]\d+)?)\s*\$/, currency: 'USD' },
      { regex: /\$\s*(\d+(?:[.,]\d+)?)/, currency: 'USD' },
      { regex: /(\d+(?:[.,]\d+)?)\s*USD/i, currency: 'USD' },
      { regex: /(\d+(?:[.,]\d+)?)\s*PLN/i, currency: 'PLN' },
      { regex: /(\d+(?:[.,]\d+)?)\s*zł/i, currency: 'PLN' },
      { regex: /(\d+(?:[.,]\d+)?)\s*SEK/i, currency: 'SEK' },
      { regex: /(\d+(?:[.,]\d+)?)\s*DKK/i, currency: 'DKK' },
      { regex: /(\d+(?:[.,]\d+)?)\s*NOK/i, currency: 'NOK' },
      { regex: /(\d+(?:[.,]\d+)?)\s*CZK/i, currency: 'CZK' },
      { regex: /(\d+(?:[.,]\d+)?)\s*Kč/i, currency: 'CZK' },
      { regex: /(\d+(?:[.,]\d+)?)\s*HUF/i, currency: 'HUF' },
      { regex: /(\d+(?:[.,]\d+)?)\s*Ft/i, currency: 'HUF' },
      // EUR patterns (check after other currencies)
      { regex: /(\d+(?:[.,]\d+)?)\s*€/, currency: 'EUR' },
      { regex: /€\s*(\d+(?:[.,]\d+)?)/, currency: 'EUR' },
      { regex: /(\d+(?:[.,]\d+)?)\s*EUR\b/i, currency: 'EUR' },
      { regex: /(\d+(?:[.,]\d+)?)\s*Euro\b/i, currency: 'EUR' },
    ];
    
    for (const p of patterns) {
      const match = cleanText.match(p.regex);
      if (match) {
        return { price: parseNum(match[1]), currency: p.currency };
      }
    }
    
    // DEFAULT: If just a number (with optional "VB" for negotiable), assume EUR
    const numMatch = cleanText.match(/(\d+(?:[.,]\d+)?)/);
    if (numMatch) {
      return { price: parseNum(numMatch[1]), currency: 'EUR' };
    }
    
    return { price: null, currency: 'EUR' };
  }
  
  try {
    const response = await fetch(url, { credentials: 'include' });
    const html = await response.text();
    
    const listing = { url, sold: false };
    
    // Check if SOLD (VERKAUFT) - mark but don't skip
    if (html.includes('VERKAUFT') || html.includes('Verkauft') || 
        html.includes('>Sold<') || html.includes('>sold<')) {
      listing.sold = true;
      console.log('Marked as SOLD:', url);
    }
    
    // SELLER: From URL /Benutzer/USERNAME/Souk
    const sellerMatch = url.match(/\/Benutzer\/([^\/]+)\/Souk/);
    listing.seller = sellerMatch ? decodeURIComponent(sellerMatch[1]) : '';
    
    // PERFUME NAME: <a href="/Parfums/..." class="black">Name</a>
    const nameMatch = html.match(/<a\s+href="[^"]*\/Parfums\/[^"]+"\s+class="black">([^<]+)<\/a>/i);
    if (nameMatch) {
      listing.name = nameMatch[1].trim();
    } else {
      const titleMatch = html.match(/<title>[^|]+\|\s*([^|]+)\s*\|/);
      listing.name = titleMatch ? titleMatch[1].trim() : '';
    }
    
    // BRAND
    const brandMatch = html.match(/<a\s+href="[^"]*\/Parfums\/([^"\/]+)"\s+class="lightgrey">([^<]+)<\/a>/i);
    listing.brand = brandMatch ? brandMatch[2].trim() : '';
    
    // COUNTRY
    const countryMatch = html.match(/flags\/([a-z]{2})\.png/i);
    listing.country = countryMatch ? countryMatch[1].toUpperCase() : '';

    // DATE ADDED: Look for "Eingestellt" or "Online seit" row
    const dateMatch = html.match(/(?:Eingestellt|Online seit|Added|Listed)<\/td>\s*<td[^>]*>([^<]+)</i);
    if (dateMatch) {
      const dateText = dateMatch[1].trim();
      // Parse German date format: "15.01.2025" or "15. Januar 2025"
      const ddmmyyyy = dateText.match(/(\d{1,2})\.(\d{1,2})\.(\d{4})/);
      if (ddmmyyyy) {
        listing.dateAdded = `${ddmmyyyy[3]}-${ddmmyyyy[2].padStart(2,'0')}-${ddmmyyyy[1].padStart(2,'0')}`;
      } else {
        // Try parsing month names
        const months = { 'januar': '01', 'februar': '02', 'märz': '03', 'april': '04', 'mai': '05', 'juni': '06',
                        'juli': '07', 'august': '08', 'september': '09', 'oktober': '10', 'november': '11', 'dezember': '12',
                        'january': '01', 'february': '02', 'march': '03', 'may': '05', 'june': '06',
                        'july': '07', 'october': '10', 'december': '12' };
        const monthMatch = dateText.match(/(\d{1,2})\.?\s*(\w+)\s*(\d{4})/i);
        if (monthMatch) {
          const month = months[monthMatch[2].toLowerCase()] || '01';
          listing.dateAdded = `${monthMatch[3]}-${month}-${monthMatch[1].padStart(2,'0')}`;
        }
      }
    }
    
    // TYPE
    if (html.includes('Flakon') && !html.includes('Probe') && !html.includes('Abfüllung')) {
      listing.type = 'bottle';
    } else {
      listing.type = 'decant';
    }
    
    // VOLUME: "35 / 100 ml" or "5 ml"
    let volMatch = html.match(/(\d+(?:[.,]\d+)?)\s*\/\s*\d+(?:[.,]\d+)?\s*ml/i);
    if (volMatch) {
      listing.volume = parseNum(volMatch[1]);
    } else {
      volMatch = html.match(/fill-circle[^>]*>(\d+(?:[.,]\d+)?)\s*ml/i);
      if (volMatch) {
        listing.volume = parseNum(volMatch[1]);
      } else {
        volMatch = html.match(/(\d+(?:[.,]\d+)?)\s*ml\b/i);
        listing.volume = volMatch ? parseNum(volMatch[1]) : null;
      }
    }
    
    // PRICE: From Preisvorstellung row (with currency detection)
    // IMPORTANT: Some users list price as "per ml" (e.g., "2,50€ pro ml")
    const priceRowMatch = html.match(/Preisvorstellung<\/td>\s*<td[^>]*>([^<]+)</i);
    if (priceRowMatch) {
      const priceText = priceRowMatch[1].trim();
      
      // Check if this is a "per ml" price
      // Patterns: /ml, pro ml, per ml, je ml, p.ml, €/ml
      const isPerMl = /\/\s*ml|pro\s*ml|per\s*ml|je\s*ml|p\.?\s*ml/i.test(priceText);
      
      const { price, currency } = parsePriceWithCurrency(priceText);
      listing.currency = currency;
      
      // Convert to EUR
      const rate = currencyRates[currency] || 1;
      
      if (isPerMl && price) {
        // Price is ALREADY per ml - don't divide by volume!
        listing.pricePerMl = price;
        listing.pricePerMlEur = Math.round(price * rate * 100) / 100;
        listing.isPerMlPrice = true;
        
        // Calculate total price from per-ml price
        if (listing.volume) {
          listing.price = Math.round(price * listing.volume * 100) / 100;
          listing.priceEur = Math.round(listing.price * rate * 100) / 100;
        }
        
        console.log('Detected PER-ML price:', priceText, '→', listing.pricePerMlEur, '€/ml');
      } else {
        // Normal total price
        listing.price = price;
        listing.priceEur = price ? Math.round(price * rate * 100) / 100 : null;
        listing.isPerMlPrice = false;
      }
    }
    
    // SHIPPING
    const shippingMatch = html.match(/Versand<\/td>\s*<td[^>]*>([^<]+)</i);
    if (shippingMatch) {
      const shipText = shippingMatch[1].trim().toLowerCase();
      if (shipText.includes('inklusiv') || shipText.includes('inklusive')) {
        listing.shipping = 0;
        listing.shippingText = 'inkl.';
      } else {
        const { price: shipPrice, currency: shipCurrency } = parsePriceWithCurrency(shippingMatch[1]);
        if (shipPrice) {
          const rate = currencyRates[shipCurrency] || 1;
          listing.shipping = Math.round(shipPrice * rate * 100) / 100;
          listing.shippingText = listing.shipping + '€';
          listing.shippingOriginal = shipPrice + ' ' + shipCurrency;
        } else {
          listing.shippingText = shippingMatch[1].trim().substring(0, 20);
        }
      }
    }
    
    // CONDITION
    const condMatch = html.match(/Zustand<\/td>\s*<td[^>]*>([^<]+)</i);
    listing.condition = condMatch ? condMatch[1].trim() : '';
    
    // Calculate price per ml (in EUR for comparison)
    // Skip if already calculated from per-ml price
    if (!listing.isPerMlPrice && listing.priceEur && listing.volume && listing.volume > 0) {
      listing.pricePerMlEur = Math.round((listing.priceEur / listing.volume) * 100) / 100;
    }
    
    // Also keep original price per ml for display
    if (!listing.isPerMlPrice && listing.price && listing.volume && listing.volume > 0) {
      listing.pricePerMl = Math.round((listing.price / listing.volume) * 100) / 100;
    }
    
    // Debug log
    const rateSource = liveRates ? 'LIVE' : 'FALLBACK';
    console.log(`[${rateSource}] Extracted:`, listing.seller, '|', listing.name, '|', 
                listing.volume, 'ml |', listing.price, listing.currency, '→', listing.pricePerMlEur, '€/ml', 
                listing.sold ? '| VERKAUFT' : '');
    
    return listing;
    
  } catch (e) {
    console.error('Error fetching:', url, e);
    return null;
  }
}
